/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2025 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.*;

/**
 * This sample explains how to add static elements to the report, like boxes, lines, labels, ... .
 */
public class StaticElementsSample extends RDCSample {

    /**
     * Adds static elements like boxes, lines, labels to a new report.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {

        String picturePath = "samples/rdc/cc_logo.png";

        try {
            //create a new Engine
            Engine eng = RDC.createEmptyEngine( exportFmt );

            //fill the engine
            //get an area and a section to fill in some elements
            Area area = eng.getArea( 2 );
            Section section = area.getSection( 0 );

            eng.setReportTitle( "static elements example" );

            /* +++ box examples +++ */

            //this adds a red box with a blue outline on (100,100)
            //with a width and a height of 1000
            //and the width of the outline is set to 40 twips (twice of normal)
            Box box = section.addBox( 100, 100, 1000, 1000 );
            box.setBackColor( RDC.COLOR_RED );
            box.setForeColor( RDC.COLOR_BLUE );
            box.setLineWidth( 40 );
            box.setCornerEllipseWidth( 2000 );
            box.setCornerEllipseHeight( 2000 );
            //this adds a box, too
            Box box2 = section.addBox( 10110, 5000, 1000, 1000 );
            box2.setBackColor( RDC.COLOR_RED );
            box2.setForeColor( RDC.COLOR_BLUE );
            box2.setLineWidth( 40 );
            box2.setCornerEllipseWidth( 200 );
            box2.setCornerEllipseHeight( 200 );
            //a box that will end in the page footer section
            Area pfArea = eng.getArea( 3 );
            Section pfSec = pfArea.getSection( 0 );
            Box box3 = section.addBox( 100, 11100, 11010, 0, 1000, pfSec );
            box3.setBackColor( RDC.COLOR_RED );
            box3.setForeColor( RDC.COLOR_BLACK );
            box3.setLineWidth( 40 );

            /* +++ line examples +++ */

            //this adds a blue, horizontal line with length of 10000 on (1100,100) and line width of 40
            Line line = section.addHorizontalLine( 1100, 100, 10000 );
            line.setForeColor( RDC.COLOR_BLUE );
            line.setLineWidth( 40 );

            //this adds a blue, vertical line with length of 10000 on (6100,100) and line width of 40
            Line line2 = section.addVerticalLine( 100, 1100, 10000 );
            line2.setForeColor( RDC.COLOR_BLUE );
            line2.setLineWidth( 40 );

            //this adds a blue, vertical line on (11100,100) with end point in page footer section on 1000 and line with of 40
            Line line3 = section.addVerticalLine( 11100, 100, 1000, pfSec );
            line3.setForeColor( RDC.COLOR_BLUE );
            line3.setLineWidth( 40 );

            /* +++ picture example +++ */

            //This adds a picture
            URL pictureURL = getClass().getResource( '/' + picturePath );
            String path;
            if( pictureURL != null ) {
                path = pictureURL.getPath();
            } else {
                path = picturePath;
            }
            Picture pic = section.addPicture( 8100, 100, 2835, 375, path );
            pic.setTopLineStyle( BorderProperties.LINE_STYLE_SINGLE );
            pic.setRightLineStyle( BorderProperties.LINE_STYLE_SINGLE );

            /* +++ text example +++ */

            //This adds a Text
            Text text = section.addText( 3000, 2500, 4000, 500 );
            //This adds a new line to the text
            Paragraph para = text.addParagraph();
            //This adds a formattable text part to the line
            TextPart tPart1 = para.addTextPart( "A " );
            TextPart tPart2 = para.addTextPart( "bold " );
            TextPart tPart3 = para.addTextPart( "text and a " );
            TextPart tPart4 = para.addTextPart( "cursive " );
            TextPart tPart5 = para.addTextPart( "one." );
            tPart1.setFontSize( 14 );
            tPart2.setFontSize( 14 );
            tPart3.setFontSize( 14 );
            tPart4.setFontSize( 14 );
            tPart5.setFontSize( 14 );
            tPart2.setBold( true );
            tPart2.setFontColor( RDC.COLOR_BLUE );
            tPart4.setItalic( true );
            tPart4.setFontColor( RDC.COLOR_RED );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 1 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new StaticElementsSample().initUI();
    }
}
